//
// Shader for simple 3d text character rendering.
// Uses a static vertex buffers:
//
// Position: (0.0, 0.0) ... (1.0, 1.0) unit square
//
// Which is transformed independently (twice) by the WVP xform and UV
// constants into the position and UV coord values.
//

#include "stddefs.fxh"

//-----------------------------------------------------------------------
// Input constants
// Xform the unit-square rectangle into size and position, then project it
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

float4 		uv_mul_add;													// x = Umul, y = Vmul, z = Uadd, w = Vadd
float4		colour;															// Multiply by the texel colour for the output colour

texture diffuseTexture : TEXTURE;							// Diffuse colour in RGB, translucency in alpha

sampler2D diffuseMap : SAMPLER < bool appEdit = false; > = sampler_state
{
	FX_SAMPLERSTATE_SRGB_TEXTURE
	Texture = < diffuseTexture >;
	MinFilter = LinearMipMapLinear;
	MagFilter = Linear;
	MipFilter = Linear;
};


//-----------------------------------------------------------------------
// Vertex Shader
struct VSINPUT
{
	float2 position : POSITION;
};


struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};


VSOUTPUT VShader( VSINPUT _input )
{
	VSOUTPUT _output = ( VSOUTPUT )0;

	// Calculate UV coords from input vertex 'position'
	_output.texCoord = ( _input.position.xy * uv_mul_add.xy ) + uv_mul_add.zw;

	_output.position = mul( float4( _input.position.xy, 0.0f, 1.0f ), worldviewproj );

	return _output;
}


//-----------------------------------------------------------------------
// Fragment Shader
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};


PSOUTPUT FragmentShader( VSOUTPUT _input )
{
	PSOUTPUT _output = ( PSOUTPUT )0;

	// Read textures
	float4	diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	_output.Colour = diffuseTexColour * colour;

	return _output;
}


//-----------------------------------------------------------------------
// Technique
technique DebugText
{
	pass Pass0
	{
		ZEnable = 1;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		blendfunc=int2(SrcAlpha, OneMinusSrcAlpha);
		cullfaceenable=false;
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx VShader();
		PixelShader = compile sce_fp_rsx FragmentShader();
#endif		
	}
}


